;;   Mesure de la "tension" d'un accord, prsent sous forme d'une liste de
;; hauteurs (MIDI-pitches). Cf. [Pressnitzer:1996], cours "Battements".
;; Algorithme simplifi, par rapport  la solution classique de
;; [Terhardt:1974] (ibid.). On utilise les intervalles de Zarlino comme
;; subdivisions "justes" de l'octave, afin d'viter l'inharmonicit intrinsque
;; du temprament gal. Le rsultat final est talonn sur une chelle [0,1] --
;; c'est--dire entre [consonant,dissonant].
;;   Cette mthode est limite aux demi-tons, tandis que celle de Pressnitzer,
;; plus sophistique, intgre les quarts-de-ton. Les travaux de [Assayag,
;; Castellengo & Malherbe, 1985, <Rapport IRCAM>], eux aussi bass sur
;; l'algorithme de Terhardt, intgrent les multiphoniques non temprs.

;;   Plus grand commun diviseur de deux nombres -- algorithme d'Euclide.
;; Cf. ped.12 et [Wertz:44 sq.] ; mais aussi la primitive gdc -- qui rend
;; inutiles les trois fonctions pgcd<x> suivantes !
(defun pgcd3 (m n)
  (cond ((> m n) (pgcd3 n m))                              ;ordre des paramtres
        ((= (rem n m) 0) m)                                ;cas final
        (t (pgcd3 (rem n m) m))))                          ;rcursion
;;   Pgcd3 applique entre les couples successifs d'une liste de nombres.
(defun pgcd+ (li)
  (if (endp (cdr li)) ()
  (cons (pgcd3 (car li) (cadr li)) (pgcd+ (cdr li)))))
;;   Pgcd+ applique jusqu' rduction  un nombre unique.
;; N.B. : donne donc la fondamentale d'une liste de rapports harmoniques, ou de
;; frquences de partiels. Ex. :
;;    (pgcd* '(1 5/4 3/2)) => 1/4                          ;rapports harmoniques
;;    (pgcd* '(400 300 100 700 600 900 1300)) => 100       ;frquences
(defun pgcd* (li)
  (if (endp (cdr li)) (car li)
  (pgcd* (pgcd+ li))))

;;   Tout serait plus simple directement avec la primitive gcd :
;;    (apply #'gcd '(1 5/4 3/2)) => 1/4                    ;rapports harmoniques
;;    (apply #'gcd '(400 300 100 700 600 900 1300)) => 100 ;frquences

;;   Liste des intervalles en demi-tons sparant les notes d'une liste de
;; MIDI-pitches de la premire note de la liste. Ex. :
;;    (nts>intervs '(60 64 67)) => (0 4 7)
(defun nts>intervs (liste)
  (let ((prem (car liste)))
    (labels ((interne (li)
               (if (endp li) ()
               (cons (- (car li) prem)
                     (interne (cdr li))))))
      (interne liste))))

;;   Intervalles de Zarlino, divisant l'octave en 12 "demi-tons" (cf. [Assayag &
;; Cholleton:1995], cours "Battements", et [Asselin, 1985, <Musique et
;; temprament>, Costallat, p. 197].
;;   <n> demi-tons -->  0 1     2   3   4   5   6     7   8   9   10  11
(defparameter Zarlino '(1 16/15 9/8 6/5 5/4 4/3 45/32 3/2 8/5 5/3 7/4 15/8))

;;   Changer les intervalles d'une liste en demi-tons en intervalles de Zarlino.
;; On tient compte des doublement d'intervalles  l'(aux)'octave(s). Ex. :
;;    (dt>Zarli (nts>intervs '(60 72 79 88))) => (1 2 3 5)
;; Ce qui correspond  des "harmoniques" du Do central formant un accord
;; "Do + Double Do + Double Sol + Triple Mi"
(defun dt>Zarli (liste)
  (if (endp liste) ()
  (let ((dt (car liste)))
    (cons (* (nth (rem dt 12) Zarlino) (expt 2 (floor (/ dt 12))))
          (dt>Zarli (cdr liste))))))

;; Donc : (pgcd* (dt>Zarli (nts>intervs '(60 72 79 88)))) => 1   ;Maxi consonant
;;        (pgcd* (dt>Zarli (nts>intervs '(60 64 67 72)))) => 1/4  ;Moins conson.
;;        (pgcd* (dt>Zarli (nts>intervs '(60 61 62 63)))) => 1/120   ;Trs diss.
;;        (pgcd* (dt>Zarli (nts>intervs '(60 61 62 63 64 65 66))))=> 1/480 ;Maxi

;;   Traitement global d'une liste de notes (MIDI-pitches), et talonnage entre
;; 0.0 (consonant) et 1.0 (dissonant) en chelle logarithmique  base 2.
(defun tension (linots)
  (* (log (pgcd* (dt>Zarli (nts>intervs linots))) 2.0) -.112272625))

;; Donc : (tension '(60 72 79 88)) => 0.0                       ;Maxi consonant
;;        (tension '(60 64 67 72)) => 0.22454526                ;Moins consonant
;;        (tension '(60 61 62 63)) => 0.77545476                ;Trs dissonant
;;        (tension '(60 61 62 63 64 65 66)) => 1.0              ;Maxi dissonant

;;   Cette mthode de mesure accrot la tension pour de petits intervalles -- ce
;; qui dcoule directement de son principe, qui consiste  calculer une
;; fondamentale intgrant l'intervalle dans sa srie "harmonique". Par ex. :
;;     (tension '(60 74)) => 0.2245  ;9M
;;     (tension '(60 73)) => 0.4386  ;9m
;;     (tension '(60 71)) => 0.3368  ;7M
;;     (tension '(60 70)) => 0.2245  ;7m
;;     (tension '(60 62)) => 0.3368  ;2M
;;     (tension '(60 61)) => 0.4386  ;2m

;;   Pour la mme raison, tous les intervalles faisant partie de la srie
;; harmonique naturelle sont considrs comme nullement dissonants -- on
;; dirait "parfaitement consonants" dans une autre contexte. Par exemple :
;;    (tension '(60 72)) => 0.0      ;8ve
;;    (tension '(60 79)) => 0.0      ;12e (double 5e)
;;    (tension '(60 88)) => 0.0      ;17M (triple 3M)
;;    (tension '(60 94)) => 0.0      ;21m (triple 7m)
;;    (tension '(60 72 79 84 88 91 94 96 98 100 103)) => 0.0 ;srie "harmonique"

;;   Elle a peu de discernement pour les clusters de demi-tons, dont elle ne
;; distingue que deux catgories. Par exemple :    
;;    (tension '(60 61 62) ... '(60 61 62 63 64 65)) => 0.7754  ;Stagne et...
;;    (tension '(60 61 62 63 64 65 66) ... ... ... ) => 1.0     ;plafonne  Max.
