;;; Ce fichier peut tre charg. Il comporte, dans son corps, des exemple 
;;; valuer manuellement. La fonction pasapas, de +clinit.cl, est utilise.
;;; Version LispWorks CAPI, 05.2009

;; TiBkG4 : la fentre maximale est :view-size #@(1150 720) (horiz. - verti.)
;; MacBookPro 2009 : la fentre maximale est 1920 x 1200 (horiz. - verti.)

;;   Variables et fonction pour sortie graphique sur une rsolution verticale de
;; 186 doubles pixels & largeur 1500 pixels simples. Il faut 24 pixels de plus
;; verticalement pour CAPI (sans doute  cause de la barre de titre).

(defparameter DimFen '(1500 . 396))
(defparameter *fen*
  (capi::contain
   (make-instance 'capi::output-pane)
                  :best-width (car DimFen)
                  :best-height (cdr DimFen)
                  :title (string (gensym "Oscilloscope-"))))

;; ATTENTION LispWorks : NE PAS VALUER LA FONCTION
;;    CONTINOSC
;; qui commande les appels graphiques 
;;    ligne
;; NI LA MACRO-FONCTION PASAPAS
;; DANS UNE FENTRE EDITOR ==> ON PERD LA MAIN !
;; ==> COPIER LES LIGNES DSIRES DANS LISTENER !!!
;; ==> VALUER DANS LISTENER !!!

;; N.B. : - cliquer bouton Break pour figer le dfilement
;;        - :c 1 ou bouton Continue pour reprendre
;;        - q pour arrter dfinitivement

(defparameter abscisse 0)

(defun ligne (num &key (max 185) &aux indx)
  (setq indx (round (+ (* (/ max 2.0) num) (/ 185.0 2.0))))
  (if (> indx max) (incf indx (- max))
      (when (minusp indx) (incf indx max)))
  (capi::apply-in-pane-process *fen*;effacer le pixel prcdent  cette abscisse
                               'gp::draw-rectangle *fen*      ;et les 9 suivants
                               abscisse 0 10 (cdr DimFen)
                               :foreground :white
                               :filled t)
  (capi::apply-in-pane-process *fen*    ;dessiner nouveau pixel  cette abscisse
                               'gp::draw-rectangle *fen*
                               abscisse (* 2 indx) 1 2
                               :foreground :black
                               :filled t)
  ;LispWorks : 0.001 sec. = timeout minimum possible = trop lent
  ;(mp::process-wait-with-timeout "pour ralentir...secs" 0.001))   ;<--LispWorks
  (dotimes (n 350) (eval t)))            ;perte de temps... <-- RGLAGE VITESSE

;;   Problme LispWorks (2009, v. 5.1.1) : on perd la main en valuant
;;    (continosc (tournePRI..
;; Peut-tre  cause de dsynchronisations entre les impression de tournePRI
;; dans Listener et affichage graphique par ligne ?
;;   D'o *cmpt*, qui sera incrment uniquement par tournePRI, afin de limiter
;;  24 le nombre d'impressions dans Listener.

(defparameter *cmpt* 0)

(defmacro continosc (programme)
  (setq abscisse 0)
  (setq *cmpt* 0)                             ;en cas d'utilisation de tournePRI
  (capi::apply-in-pane-process *fen*                         ;effacer la fentre
                               'gp::draw-rectangle *fen*
                               0 0 (car DimFen) (cdr DimFen)
                               :foreground :white
                               :filled t)
  (format *debug-io* "~&Taper quelque chose, par ex. <=>, pour ~
                     commencer, et <q> pour finir.~%")
  (read-char *debug-io* nil t t)
  (terpri)
  `(loop
     ,programme
     (setq abscisse (mod (1+ abscisse) (car DimFen)))
     (when (equal #\q (read-char-no-hang *debug-io* nil t t))
       (return 'ok))))

;;   Exemples suivants avec (sin <radians>), mise en oeuvre sous la forme
;; (sin (incf <angle> <pas>)), ou (sin (incf <angle> (freq <Hz>))).
;; N.B. : 360 degrs = 2*pi radians.

;;   Calcul des <pas> angulaires  partir d'une frquence <f>, par rapport  un
;; taux d'chantillonnage de 32000 ch./sec. (plus rond que 44100...).
(defun freq (f)
  (/ (* 2.0 pi f) 32000.0))
;; N.B. : priode, en chantillons = (/ (* 2 pi) <pasN>)
;;                                 = (/ (* 2 pi) (freq <Hz>))

;;   Composante grave G : 100 Hz => 320 ch./priode
(defparameter fG 100)
(defparameter pasG (freq fG))
(defparameter angG (- (freq fG)))
(defparameter amG 0.6)
;;   Composante moyenne M : 1000 Hz => 32 ch./priode
(defparameter fM 1000) 
(defparameter pasM (freq fM))
(defparameter angM (- (freq fM)))
(defparameter amM 0.3)
;;   Composante aige A : 16000 Hz => 2 ch./priode
(defparameter fA 16000)
(defparameter pasA (freq fA))
(defparameter angA (- (/ (freq fA) 2.0)))
(defparameter amA 0.1)
(defparameter mix
  '(+ (* amG (sin (incf angG pasG)))
      (* amM (sin (incf angM pasM)))
      (* amA (sin (incf angA pasA)))))

;;   Visualisation des composantes spares
;;    (continosc (ligne (* amG (sin (incf angG pasG)))))    ;grave
;;    (continosc (ligne (* amM (sin (incf angM pasM)))))    ;moyenne
;;    (continosc (ligne (* amA (sin (incf angA pasA)))))    ;aige
;;   Visualisation de la somme des trois
;;    (continosc (ligne (eval mix)))                        ;mixage

;;; Dfinitions de classes et mthodes 2p2z

(defclass filtre () 
  ((nom :initarg :nom) (yn :initarg :yn))                          ; yn = sortie
  (:default-initargs :nom nil :yn  0  ))

;;   Algorithme 2p2z~ de Max NeXT/ISPW (Miller Puckette).
(defclass 2p2z (filtre)
  ((z0 :initarg :z0) (z1 :initarg :z1) (z2 :initarg :z2)
   (c0 :initarg :c0) (c1 :initarg :c1) (c2 :initarg :c2)
   (d0 :initarg :d0) (d1 :initarg :d1) (d2 :initarg :d2))
  (:default-initargs :nom '2p2z_de_base 
                     :z0 0 :z1 0 :z2 0
                     :c0 0 :c1 0 :c2 0
                     :d0 0 :d1 0 :d2 0))

;;   Traitement d'un chantillon <xn> par l'instance de 2p2z nomme <flt>.
(defmethod traite ((flt 2p2z) xn)
  (reinitialize-instance flt :z0
    (+ (* xn                   (slot-value flt 'c0))
       (* (slot-value flt 'z1) (slot-value flt 'c1))
       (* (slot-value flt 'z2) (slot-value flt 'c2))))
  (reinitialize-instance flt :yn
    (+ (* (slot-value flt 'z0) (slot-value flt 'd0))
       (* (slot-value flt 'z1) (slot-value flt 'd1))
       (* (slot-value flt 'z2) (slot-value flt 'd2))))
  (reinitialize-instance flt :z2 (slot-value flt 'z1))
  (reinitialize-instance flt :z1 (slot-value flt 'z0)))

;; Macro pour traiter un chantillon et afficher la ligne.
(defmacro tourne (input filtre &optional (gain 1))
  `(progn (traite ,filtre (eval ,input))
          (ligne (* ,gain (slot-value ,filtre 'yn)))))

;; Macro pour traiter un chantillon, afficher la ligne et imprimer sa valeur.
;; Pour LispWorks : limit  24 impressions par le compteur *cmpt*. Pas trs
;; lgant, mais autrement, on perd la main (cf. ci-dessus) !

(defmacro tournePRI (input filtre &optional (gain 1))
  `(progn (traite ,filtre (eval ,input))
          (ligne (* ,gain (slot-value ,filtre 'yn)))
          (when (< (incf *cmpt*) 25)
            (format t "~5,1F" (slot-value ,filtre 'yn)))
          (when (= 25 *cmpt*)
            (format t " ..."))))

;; Macro pour traiter un chantillon SANS afficher la ligne.
(defmacro tournePL (input filtre &optional (gain 1))
  `(progn (traite ,filtre (eval ,input))
          (* ,gain (slot-value ,filtre 'yn))))

;;; Dfinitions de filtres

;;   Rien ne passe ! Tous coefficients  zro.
(defparameter PasseRien
  (make-instance '2p2z))
;    (continosc (tourne mix PasseRien))

;;   Tout passe ! Coefficients c0 et d0  1.
(defparameter PasseTout
  (make-instance '2p2z :c0 1 :d0 1))
;    (continosc (tourne mix PasseTout))

;;   Filtre passe-bas simple. Exemple de [STEIGLITZ, K, 1974 : An Introduction
;; to Discrete Systems, Wiley : 50 sq.] - "moving average simple filter". Aussi
;; quivalent  [SMITH, J. O. III, 1981 : Introduction to Digital Filter Theory,
;; CCRMA : 4 sq.] - "simplest (and by no means ideal) low-pass filter".
(defparameter PasseBasSteiglitz
  (make-instance '2p2z :c0 1 :d0 0.5 :d1 0.5))
;    (continosc (tourne mix PasseBasSteiglitz))

;;   Filtre passe-haut simple. Exemple de [STEIGLITZ, op. cit. : 56-57] - second
;; exemple de "moving average simple filter".
(defparameter PasseHautSteiglitz
  (make-instance '2p2z :c0 1 :d0 0.5 :d1 -0.5))
;    (continosc (tourne mix PasseHautSteiglitz))

;;   Filtre passe-bas de 4X et Max NeXT/ISPW : frquence de coupure G. Cf. aussi
;; exemple de [MOORE, R., 1978 : "An Introduction to the Mathematics of Digital
;; Signal Processing (Part II)", CMJ (2)2 : 56] - "simplest [...] first-order
;; LPF".
(defparameter lpf0
  (make-instance '2p2z :c0 pasG :c1 (- 1 pasG) :d0 1))
;    (continosc (tourne mix lpf0))

;;   Filtre passe-haut de 4X et Max NeXT/ISPW : frquence de coupure 8000.
(defparameter hpf0
  (make-instance '2p2z
    :c0 (- 1 (freq 8000)) :c1 (- 1 (freq 8000)) :d0 1 :d1 -1))
;    (continosc (tourne mix hpf0))

;;   Filtres passe-bas et passe-haut de 4X et Max NeXT/ISPW en srie, pour effet
;; passe-bande : frquence de rsonance M.
(defparameter lpf0b
  (make-instance '2p2z :c0 pasM :c1 (- 1 pasM) :d0 1))
(defparameter hpf0b
  (make-instance '2p2z :c0 (- 1 pasM) :c1 (- 1 pasM) :d0 1 :d1 -1))
;    (continosc (tourne mix lpf0b))               ;partie passe-bas
;    (continosc (tourne mix hpf0b))               ;partie passe-haut
;   Les deux en srie : passe-haut -> passe-bas
;    (continosc (tourne (tournePL mix hpf0b) lpf0b)) ;-> passe-bande
;   Les deux en srie : passe-bas -> passe-haut
;    (continosc (tourne (tournePL mix lpf0b) hpf0b)) ;-> passe-bande

;;   Filtres 2 ples passe-bande de [SMITH,: op. cit. : 46-48] : frquence de
;; rsonance M. Coefficients de qualit 0 <= R < 1 selon les cas 2pBande-<R>.
;; Conforme aux rponses de la Fig. 23, p. 48.
(defparameter 2pBande-0.8
  (make-instance '2p2z
    :c0 1 :c1 (- (* -2 0.8 (cos pasM))) :c2 (- (expt 0.8 2)) :d0 1))
;    (continosc (tourne mix 2pBande-0.8 0.05)) ; -26dB
(defparameter 2pBande-0.85
  (make-instance '2p2z
    :c0 1 :c1 (- (* -2 0.85 (cos pasM))) :c2 (- (expt 0.85 2)) :d0 1))
;    (continosc (tourne mix 2pBande-0.85 0.025)) ; -32dB
(defparameter 2pBande-0.95
  (make-instance '2p2z
    :c0 1 :c1 (- (* -2 0.95 (cos pasM))) :c2 (- (expt 0.95 2)) :d0 1))
;    (continosc (tourne mix 2pBande-0.95 0.01)) ; -40dB

;;   Filtres 2 zros rejet-de-bande de [SMITH,: op. cit. : 49-51] : frquence de
;; rsonance M. Coefficients de qualit 0 <= R < 1 selon les cas 2zRejet-<R>.
;; Conforme aux rponses de la Fig. 25, p. 51.
(defparameter 2zRejet-0.5
  (make-instance '2p2z :c0 1 :d0 1 :d1 (* -2 0.5 (cos pasM)) :d2 (expt 0.5 2)))
;    (continosc (tourne mix 2zRejet-0.5))
(defparameter 2zRejet-0.8
  (make-instance '2p2z :c0 1 :d0 1 :d1 (* -2 0.8 (cos pasM)) :d2 (expt 0.8 2)))
;    (continosc (tourne mix 2zRejet-0.8))
(defparameter 2zRejet-0.95
  (make-instance '2p2z
    :c0 1 :d0 1 :d1 (* -2 0.95 (cos pasM)) :d2 (expt 0.95 2)))
;    (continosc (tourne mix 2zRejet-0.95))

;;   Oscillateur sinusodal sans table. Cf. [???, 1997 : "Synthesis without
;; Lookup Tables", CMJ 20:3 : 5]. L'entre du "filtre" (constante  -333) n'a
;; pas d'influence, le coefficient c0 tant  zro. Les valeurs initiales de
;; z1 et z2 sont ici les valeurs simplifies pour une phase initiale <W>.
(defparameter W (/ (* 442.0 pi) 32000.0))
;    Pour obtenir d'autre(s) frquences...
;    <1> (defparameter W (/ (* 221.0 pi) 32000.0))
;        (defparameter W (/ (* 884.0 pi) 32000.0))
;        ...
;    <2> recrer l'instance oscillateur
(defparameter oscillateur
  (make-instance '2p2z :c1 (* 2.0 (cos W))
                       :c2 -1.0
                       :d0 1.0
                       :z2 (- (sin W))       ))
;    (continosc (tourne -333 oscillateur))

;;   Puissances successives de <n>. Il faut initialiser une instance avec
;; c1=<n>, d0=1, z1=1. L'entre du "filtre" (constante  -333) n'a pas
;; d'influence, le coefficient c0 tant  zro.
(defparameter 2**n
  (make-instance '2p2z :c1 2 :d0 1 :z1 1))
;    (pasapas (print (prog2 (traite 2**n -333) (slot-value 2**n 'yn))))
(defparameter 3**n
  (make-instance '2p2z :c1 3 :d0 1 :z1 1))
;    (pasapas (print (prog2 (traite 3**n -333) (slot-value 3**n 'yn))))
(defparameter 5**n
  (make-instance '2p2z :c1 5 :d0 1 :z1 1))
;    (pasapas (print (prog2 (traite 5**n -333) (slot-value 5**n 'yn))))

;;   Intgrateur simple. Cf. fonction integ, Chap. 12, prsente comme utile
;; pour convertir une liste d'intervalles de temps en une liste de dates de
;; dpart pour une succession de notes CSound :
;;    ? (setq intervs '(0.5 1 2 1.0 10 1.5 2.3))
;;    (0.5 1 2 1.0 10 1.5 2.3)
;;    ? (integ intervs)
;;    (0 0.5 1.5 3.5 4.5 14.5 16.0 18.3)
(defparameter integrateur
  (make-instance '2p2z :c0 1 :c1 1 :d0 1))
;; On obtient le mme rsultat ci-dessous...
;    (let ((intervs '(0.5 1 2 1.0 10 1.5 2.3))) (pasapas (print (prog2 (traite integrateur (pop intervs)) (slot-value integrateur 'yn)))))

;;   Srie de Fibonacci : F(n) = F(n-1) + F(n-2).
;; L'entre du "filtre" (constante  -333) n'a pas d'influence, le coefficient
;; c0 tant  zro. La srie de Fibonacci dbute sans le premier 1 : on obtient
;;    1, 2, 3, 5, 8, 13, 21, 34, 55...
(defparameter fibonacci
  (make-instance '2p2z  :c1 1 :c2 1 :d0 1 :z0 1 :z1 1))
;    (pasapas (print (prog2 (traite fibonacci -333) (slot-value fibonacci 'yn))))

;;   Variante 1 de Fibonacci. En posant d1=-1, on fait intervenir un "zro" du
;; "filtre". L'entre du "filtre" (constante  -333) n'a pas d'influence, le
;; coefficient c0 tant  zro. On obtient une srie de Fibonacci complte,
;; prcde d'un 0 : 0, 1, 1, 2, 3, 5, 8, 13, 21, 34, 55...
(defparameter fibonacciV1
  (make-instance '2p2z  :c1 1 :c2 1 :d0 1 :d1 -1 :z0 1 :z1 1))
;    (pasapas (print (prog2 (traite fibonacciV1 -333) (slot-value fibonacciV1 'yn))))

;;   Variante 2 de Fibonacci. En posant d1=-1 et d2=1, on fait intervenir deux
;; "zros" du "filtre". L'entre du "filtre" (constante  -333) n'a pas
;; d'influence, le coefficient c0 tant  zro. On obtient une srie de
;; Fibonacci diffrente, prcde d'un 0 : 0, 2, 2, 4, 6, 10, 16...
(defparameter fibonacciV2
  (make-instance '2p2z  :c1 1 :c2 1 :d0 1 :d1 -1 :d2 1 :z0 1 :z1 1))
;    (pasapas (print (prog2 (traite fibonacciV2 -333) (slot-value fibonacciV2 'yn))))

;;   Variante 3 de Fibonacci. En posant d1=1 et d2=-1, on fait intervenir deux
;; "zros" du "filtre". L'entre du "filtre" (constante  -333) n'a pas
;; d'influence, le coefficient c0 tant  zro. On obtient la mme srie de
;; Fibonacci que la variante 2, mais sans le 0 initial  : 2, 2, 4, 6, 10...
(defparameter fibonacciV3
  (make-instance '2p2z  :c1 1 :c2 1 :d0 1 :d1 1 :d2 -1 :z0 1 :z1 1))
;    (pasapas (print (prog2 (traite fibonacciV3 -333) (slot-value fibonacciV3 'yn))))

;;   Variante 4 de Fibonacci. Sans "zros" : seul c2 est chang  -1. L'entre
;; du "filtre" (constante  -333) n'a pas d'influence, le coefficient c0 tant 
;; zro. La sortie oscille priodiquement entre trois valeurs sur une priode de
;; six chantillons. Une attnuation est  applique pour rendre cette priode
;; plus visible.
(defparameter fibonacciV4
  (make-instance '2p2z  :c1 1 :c2 -1 :d0 1 :z0 1 :z1 1))
;    (continosc (tournePRI -333 fibonacciV4 0.05))

;;   Variante 5 de Fibonacci. Sans "zros" : c1 et c2 sont changs  -1.
;; L'entre du "filtre" (constante  -333) n'a pas d'influence, le coefficient
;; c0 tant  zro. La sortie oscille priodiquement entre trois valeurs sur une
;;  priode de trois chantillons. Une attnuation est  applique pour rendre
;; cette priode plus visible.
(defparameter fibonacciV5
  (make-instance '2p2z  :c1 -1 :c2 -1 :d0 1 :z0 1 :z1 1))
;    (continosc (tournePRI -333 fibonacciV5 0.05))

;;   Variante 6 de Fibonacci. Sans "zros" : c1=-1 et c2=1. L'entre du "filtre"
;; (constante  -333) n'a pas d'influence, le coefficient c0 tant  zro. On
;; obtient les termes de la srie de Fibonacci (sans le premier 1, cf. fibonacci
;; ci-dessus), MAIS avec des signes alterns : -1, 2, -3, 5, -8...
(defparameter fibonacciV6
  (make-instance '2p2z  :c1 -1 :c2 1 :d0 1 :z0 1 :z1 1))
;    (pasapas (print (prog2 (traite fibonacciV6 -333) (slot-value fibonacciV6 'yn))))

;;   Variante 7 de Fibonacci. Un "zro" ajout  la variante n4 : d1=1. Oscille
;; priodiquement entre cinq valeurs sur une priode de six chantillons,
;; diffrents de ceux de la n4.
(defparameter fibonacciV7
  (make-instance '2p2z  :c1 1 :c2 -1 :d0 1 :d1 1 :z0 1 :z1 1))
;    (continosc (tournePRI -333 fibonacciV7 0.05))

;;   Variante 8 de Fibonacci. Deux "zros" ajouts  la variante n4 : d1=1,
;; d2=1. La sortie oscille priodiquement entre trois valeurs sur une priode de
;; six chantillons, cf. n4.
(defparameter fibonacciV8
  (make-instance '2p2z  :c1 1 :c2 -1 :d0 1 :d1 1 :d2 1 :z0 1 :z1 1))
;    (continosc (tournePRI -333 fibonacciV8 0.05))

;;   Variante 9 de Fibonacci. Deux "zros" ajouts  la variante n4 : d1=1,
;; d2=-1. La sortie oscille priodiquement entre trois valeurs sur une priode
;; de six chantillons., cf. n4.
(defparameter fibonacciV9
  (make-instance '2p2z  :c1 1 :c2 -1 :d0 1 :d1 1 :d2 -1 :z0 1 :z1 1))
;    (continosc (tournePRI -333 fibonacciV9 0.05))

;;   Variante 10 de Fibonacci. Deux "zros" ajouts  la variante n4 : d1=-1,
;; d2=-1. La sortie oscille priodiquement entre trois valeurs sur une priode
;; de six chantillons., cf. n4.
(defparameter fibonacciV10
  (make-instance '2p2z  :c1 1 :c2 -1 :d0 1 :d1 -1 :d2 -1 :z0 1 :z1 1))
;    (continosc (tournePRI -333 fibonacciV10 0.05))

;;   Variante 11 de Fibonacci. Deux "zros" ajouts  la variante n4 : d1=-1,
;; d2=1. Reste  0.
(defparameter fibonacciV11
  (make-instance '2p2z  :c1 1 :c2 -1 :d0 1 :d1 -1 :d2 1 :z0 1 :z1 1))
;    (continosc (tourne -333 fibonacciV11))

;;   Variante 12 de Fibonacci. Un "zro" ajout  la variante n5 : d1=1.
;; Oscille priodiquement entre trois valeurs sur une priode de trois
;; chantillons, cf. n 5, mais avec "dents de scie" inverses.
(defparameter fibonacciV12
  (make-instance '2p2z  :c1 -1 :c2 -1 :d0 1 :d1 1 :z0 1 :z1 1))
;    (continosc (tournePRI -333 fibonacciV12 0.05))

;;   Variante 13 de Fibonacci. Deux "zros" ajouts  la variante n5 : d1=-1,
;; d2=-1. Oscille priodiquement entre trois valeurs sur une priode de trois
;; chantillons, cf. n 5.
(defparameter fibonacciV13
  (make-instance '2p2z  :c1 -1 :c2 -1 :d0 1 :d1 -1 :d2 -1 :z0 1 :z1 1))
;    (continosc (tournePRI -333 fibonacciV13 0.05))

;;   Variante 14 de Fibonacci. Deux "zros" ajouts  la variante n5 : d1=1,
;; d2=-1. Oscille priodiquement entre trois valeurs sur une priode de trois
;; chantillons, cf. n 12.
(defparameter fibonacciV14
  (make-instance '2p2z  :c1 -1 :c2 -1 :d0 1 :d1 1 :d2 -1 :z0 1 :z1 1))
;    (continosc (tournePRI -333 fibonacciV14 0.05))

;;   Variante 15 de Fibonacci. Deux "zros" ajouts  la variante n5 : d1=-1,
;; d2=1. Oscille priodiquement entre trois valeurs sur une priode de trois
;; chantillons, cf. n 12.
(defparameter fibonacciV15
  (make-instance '2p2z  :c1 -1 :c2 -1 :d0 1 :d1 -1 :d2 1 :z0 1 :z1 1))
;    (continosc (tournePRI -333 fibonacciV15 0.05))

;;   Variante 16 de Fibonacci. Deux "zros" ajouts  la variante n5 : d1=1,
;; d2=1. Reste  0, cf. n11.
(defparameter fibonacciV16
  (make-instance '2p2z  :c1 -1 :c2 -1 :d0 1 :d1 1 :d2 1 :z0 1 :z1 1))
;    (continosc (tourne -333 fibonacciV16))

;; Etc...

;;; Variante 3p3z...

;;   Algorithme 3p3z dduit du prcdent
(defclass 3p3z (2p2z)
  ((z3 :initarg :z3) (c3 :initarg :c3) (d3 :initarg :d3))
  (:default-initargs :z3 0 :c3 0 :d3 0))

;;   Traitement d'un chantillon <xn> par l'instance de 3p3z nomme <flt>.
(defmethod traite ((flt 3p3z) xn)
  (reinitialize-instance flt :z0
    (+ (* xn                   (slot-value flt 'c0))
       (* (slot-value flt 'z1) (slot-value flt 'c1))
       (* (slot-value flt 'z2) (slot-value flt 'c2))
       (* (slot-value flt 'z3) (slot-value flt 'c3))))
  (reinitialize-instance flt :yn
    (+ (* (slot-value flt 'z0) (slot-value flt 'd0))
       (* (slot-value flt 'z1) (slot-value flt 'd1))
       (* (slot-value flt 'z2) (slot-value flt 'd2))
       (* (slot-value flt 'z3) (slot-value flt 'd3))))
  (reinitialize-instance flt :z3 (slot-value flt 'z2))
  (reinitialize-instance flt :z2 (slot-value flt 'z1))
  (reinitialize-instance flt :z1 (slot-value flt 'z0)))

;;   Srie de Padovan : P(n) = P(n-2) + P(n-3).
;; L'entre du "filtre" (constante  -333) n'a pas d'influence, le coefficient
;; c0 tant  zro. La srie de Padovan dbute sans les deux premiers 1 : on
;; obtient
;;    1, 2, 2, 3, 4, 5, 7, 9, 12, 16, 21, 28, 37...
(defparameter padovan
  (make-instance '3p3z  :c1 0 :c2 1 :c3 1 :d0 1 :z0 1 :z1 1 :z2 1))
;    (pasapas (print (prog2 (traite padovan -333) (slot-value padovan 'yn))))

;;   Srie des vaches de Narayana : N(n) = N(n-1) + N(n-3).
;; L'entre du "filtre" (constante  -333) n'a pas d'influence, le coefficient
;; c0 tant  zro. La srie dbute sans ses deux premiers 1 : on obtient
;;    1, 2, 3, 4, 6, 9, 13, 19, 28, 41...
(defparameter narayana
  (make-instance '3p3z  :c1 1 :c2 0 :c3 1 :d0 1 :z0 1 :z1 1 :z2 1))
;    (pasapas (print (prog2 (traite narayana -333) (slot-value narayana 'yn))))

;;   Srie XXX1 : X(n) = X(n-1) + X(n-2) + X(n-3).
;; L'entre du "filtre" (constante  -333) n'a pas d'influence, le coefficient
;; c0 tant  zro. La srie dbute sans ses trois premiers 1 : on obtient
;;    3, 5, 9, 17, 31, 57, 105, 193, 355, 653...
(defparameter XXX1
  (make-instance '3p3z  :c1 1 :c2 1 :c3 1 :d0 1 :z0 1 :z1 1 :z2 1 :z3 1))
;    (pasapas (print (prog2 (traite XXX1 -333) (slot-value XXX1 'yn))))

;;   Variante XXX-2 : X(n) = -2*X(n-1) + X(n-2) + X(n-3).
;; L'entre du "filtre" (constante  -333) n'a pas d'influence, le coefficient
;; c0 tant  zro. La srie dbute sans ses trois premiers 1 : on obtient
;;    0, 2, -3, 8, -17, 39, -87, 196, -440, 989, -2222, 4993...
(defparameter XXX-2
  (make-instance '3p3z  :c1 -2 :c2 1 :c3 1 :d0 1 :z0 1 :z1 1 :z2 1 :z3 1))
;    (pasapas (print (prog2 (traite XXX-2 -333) (slot-value XXX-2 'yn))))

;; Etc...
